unit exelogu;

interface
uses Classes, SysUtils;

type
  TExeLogLog = procedure(const msg:string) of object;

  TLogEvents = (
    leInfo,   { default event, information about anything happening, used if Log() method used }
    leAction, { some important not very often occuring program process started/ended }
    leDebug,  { debugging purposes }
    leUserWarning, { user tried to do something what he should not, gui window appeared }
    leUserInfo,    { user is informed that some action started, is going on or ended,
                     gui window appeared}
    leUserConfirm,  { user is aked to do some decision, gui window appeared }
    leWarning,     { some expectable problem arised (disk full,...) }
    leError,       { some real program error occurred (unexpected, that should not happen) }
    leFatalError   { real disaster happened }
  );

  TLogEvent = set of TLogEvents;

  TExeLog = class(TComponent)
  private
    FLogDir: string;
      { Directory for log files (if not absolute path, then relative to exedir
        is considered), default = LOG. Used if FFileName does not contain absolute
        path }
    FFileName: string;
      { Name of the log file. If none specified then ExeName.Log is used.
        If does not contain absolute path then placed to FLogDir. }{logu}
    FText: text;
      { the text file handle }
    FActive: boolean;
      { is the log file opened }
    FAutoOpen: boolean;
      { should the log file be opened upon program start? }
    FOverWrite: boolean;
      { should erase eventually existing log file upon opening? }
    FLogEvent: TLogEvent;
      { last log event logged }
    FAutoFlushInterval: integer;
      { how often the log should be flushed to disk (in seconds), not
        flushed at all if = 0 (flushing can take place only during Log call) }
    FLastFlushTime: TDateTime;
      { last time when the file was flushed to disk }
    FOnLog: TExeLogLog;
    {v0.26}
    FAssigningOnLog: boolean;
    {/v0.26}
  protected
    procedure SetActive(OnOff:boolean);
    procedure SetFileName(const AFileName: string);
    {v0.26}
    procedure SetOnLog(AExeLogLog: TExeLogLog);
    {/v0.26}
  public
    constructor Create(AOwner: TComponent; const AFileName: string);reintroduce;
      { if AFileName = '' then ExeName.LOG will be used }
    destructor Destroy;override;
    procedure Log(const Msg: string);
    procedure LogEvent(ALogEvent: TLogEvent; const Msg: string);

    property Active: boolean read FActive write SetActive;
    property AutoOpen: boolean read FAutoOpen write FAutoOpen;
    property FileName: string read FFileName write SetFileName;
    property OverWrite: boolean read FOverWrite write FOverWrite;
    property AutoFlushInterval: integer read FAutoFlushInterval write FAutoFlushInterval;
    property OnLog: TExeLogLog read FOnLog write {v0.26}SetOnLog{/v0.26 FOnLog};
    property LogDir: string read FLogDir write FLogDir;
  end;

const
  ExeLog: TExeLog = nil;

implementation

procedure TExeLog.SetActive(OnOff:boolean);
begin
  if OnOff <> FActive then begin
    if OnOff then begin
      AssignFile(FText, FFileName);
      if FileExists(FFileName) then begin
        if FOverWrite then
          Rewrite(FText)
        else
          Append(FText);
      end else begin
        Rewrite(FText);
      end;
      Writeln(FText, '');
      Writeln(FText, '*** Log Started ' + FormatDateTime('',Now) + ' ***');
      Writeln(FText, '');
    end else begin
      Writeln(FText, '');
      Writeln(FText, '*** Log Stopped ' + FormatDateTime('',Now) + ' ***');
      Writeln(FText, '');
      CloseFile(FText);
    end;
    FActive := OnOff;
  end;
end;

procedure TExeLog.SetFileName(const AFileName: string);
var wasa:boolean;
begin
  if AFileName = '' then
    exit;
  if AFileName <> FFileName then begin
    wasa := Active;
    if wasa then
      Active := false;
    FFileName := ExpandFileName(AFileName);
    if wasa then
      Active := true;
  end;
end;

constructor TExeLog.Create(AOwner: TComponent; const AFileName: string);
begin
  inherited Create(AOwner);
  if AFileName = '' then begin
    SetFileName(ChangeFileExt(paramstr(0), '.LOG'));
  end;
end;

destructor TExeLog.Destroy;
begin
  Active := false;
  inherited Destroy;
end;

procedure TExeLog.Log(const Msg: string);
begin
  if not Active then
    exit;
  LogEvent([leInfo], Msg);
end;

procedure TExeLog.SetOnLog(AExeLogLog: TExeLogLog);
begin
{  if AExeLogLog <> FOnLog then
    exit;}
  fAssigningOnLog := true;
  try
    FOnLog := AExeLogLog;
  finally
    fAssigningOnLog := false;
  end;
end;

procedure TExeLog.LogEvent(ALogEvent: TLogEvent; const Msg: string);
begin
  if not Active then
    exit;
  FLogEvent := ALogEvent;
  Writeln(FText, Msg);
  if {v0.26} (not FAssigningOnLog) and {v0.26} Assigned(FOnLog) then
    FOnLog(Msg);
  if FAutoFlushInterval <> 0 then begin
    if (Now - FLastFlushTime) * 24 * 3600 > FAutoFlushInterval then begin
      Flush(FText);
      FLastFlushTime := Now;
    end;
  end;
end;

initialization
  ExeLog := TExeLog.Create(nil, '');
  {ExeLog.OverWrite := true;}
  ExeLog.Active := true;
finalization
  ExeLog.Free;
end.
