unit Language;
{$DEFINE USESRC}
interface
(* Multilanguage support for Delphi application.

  Usage:

  - put Language component on the main form
    ( or include in MainForm.FormCreate method:
        "LanguageInit('');"  or   "LanguageInit('xxx.Lng');" )

    ( - if you assigned Screen.OnActiveFormChange handler, call from it also
        LanguageCheckForms procedure )

  - in your source code use GetTxt function for assigning language dependent
    strings to variables (parameter to this function is text string in your
    development language, preferably english)

  - if you are using text constants, put before every such constant that
    should be included in .lng file {#}, e.g.:

      {#}'This is in english, will be included in .lng file';


  Creating/Using .LNG files:

  - Set Language.RegisterMode true during application development, AppName.LNG
    file will then be created (and updated) for all texts in forms that get
    created (each form has its section in LNG file) and for all parameters
    of GetTxt calls used (all these texts are in [Misc] section in LNG file).

    If the LNG file did not exist, than also all *.pas source files will be
    scanned for strings starting with {#} tag (placed also to [Misc] section in
    LNG file). This is done only IFDEF USESRC (FileScanner component is needed
    for that)

    Components or properties with name starting with '_' char will not be included
    to .LNG files

  - .lng texts can then be modified, .lng file renamed and its name can be
    assigned to Language.FileName property (designtime or runtime)

  Limitation:

  '=' sign can not be used in constant strings (GetTxt parameters) due to
  the INI structure of .LNG files
*)

uses
  Windows, Messages, SysUtils, Classes,
  {$IFNDEF CONSOLE}
  Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls,
  {$ENDIF}
  IniFiles, TypInfo {$IFDEF USESRC}, FileScanner{$ENDIF}
  ,Compareu;

const
  LngExt = '.lng';
  MiscSec = 'Misc';
  ExcludePrefix = '_';
    { properties or components with names starting with this prefix won't be
      changed by Language }
  SourceTextTag = '{#}';
    { put this tag before every string constant (i.e. just before ') in the
      source files, that should be included in the language file }

type
  TRegisterText = procedure(const ASection: string; const ATextName: string; const ATextValue: string) of object;
  TGetText = function(const ASection: string; const ATextName: string; const ADefValue: string): string of object;

  {TStringObj = class(TObject)
    Text: string;
    constructor Create(const AText: string);reintroduce;
  end;}

  TLanguage = class(TComponent)
  private
    { Private declarations }
    FActive: boolean;
    FRegisterMode: boolean;
    {FUseStrings: boolean;
    FStrings: TStringList;}
    FIniFile: TMemIniFile;
    FOnRegisterText: TRegisterText;
    FOnGetText: TGetText;
    FFileName: string;{subclasser}
    {FOldWndMethod : TWndMethod;
    procedure SubClassWndProc(var Message: TMessage);
    procedure UpdateOwner(AOwner: TComponent);}
    FCheckedForms: TList;
    FAllTextProps: boolean;
    FAlwaysScanSrcFiles: boolean;
    {v0.47}
    FInspectAllForms: boolean;
      { if false, then checks only forms with Owner = Application; otherwise
        all (but those with classnames excluded) }
    {/v0.47}
  protected
    { Protected declarations }
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure CheckComponent(const ASection: string; const APathName: string; c: TComponent);
    procedure Load;{ texts from file to memory }
    procedure Save;{ texts from memory to file }
    procedure SetFileName(const AFileName: string);
    procedure SetLanguageName(const AName: string);
    function LoadIniFile: TMemIniFile;
    {$IFNDEF CONSOLE}
    procedure ActiveFormChange(Sender: TObject);
    {$ENDIF}
    {procedure AfterConstruction; override;
    procedure BeforeDestruction; override;}
    {$IFDEF USESRC}
    procedure DoSrcLine(const ALine: string; var AResult: TDoLineResult);
    {$ENDIF}
    function GetCheckedForms: TList;
    function GetLanguageName: string;
    {v0.44}
    function IsFormClassToIgnore(const AClassName: string): boolean;
    {/v0.44}
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy;override;
    {$IFDEF USESRC}
    procedure ScanSourceFiles;
    {$ENDIF}
    {$IFNDEF CONSOLE}
    procedure Check;
      { Called once after load }
    procedure CheckForms;
      { Call periodically runtime (from Screen.OnActiveFormChange) }
    procedure CheckForm(AForm: TForm);
      { Called from CheckForms. Can be called explicitly before showing form
        to speed up }
    {$ENDIF}

    procedure RegisterText(const ASection: string; const ATextName: string; const ATextValue: string);
      { Called for every text name/value found in components or passed to
        GetTxt function runtime (if in register mode). If not found in text
        storage, will add it there (the OnRegisterText event must do it) }
    function GetText(const ASection: string; const ATextName: string; const ADefValue: string): string;
      { If Active, then this method called from every GetTxt function,
        returns: if not ATextName found storage, returns ADefValue, if ADefValue
          = '', then returns ATextName }
    property CheckedForms: TList read GetCheckedForms;
    property LanguageName: string read GetLanguageName;
  published
    { Published declarations }
    property Active: boolean read FActive write FActive;
    property RegisterMode: boolean read FRegisterMode write FRegisterMode;
    property OnGetText: TGetText read FOnGetText write FOnGetText;
    property OnRegisterText: TRegisterText read FOnRegisterText write FOnRegisterText;
    property FileName: string read FFileName write SetFileName;
      { LNG file name }
    property AllTextProps: boolean read FAllTextProps write FAllTextProps;
      { If false only properties that has Caption, Hint, Label, or Title substring in
        their name will be included in LNG file,
        if true all properties of tkString, tkLString and tkWString kind }
    property AlwaysScanSrcFiles: boolean read FAlwaysScanSrcFiles write FAlwaysScanSrcFiles;
    {v0.47}
    property InspectAllForms: boolean read FInspectAllForms write FInspectAllForms;
    {/v0.47}
  end;

procedure Register;

function GetTxt(const AText: string): string;

procedure LanguageInit(const AFileName: string);
  { call if Language component not placed on MainForm to initialize the language
    support }
{$IFNDEF CONSOLE}
procedure LanguageCheckForms;
  { Call from Screen.ActiveFormChange handler }
{$ENDIF}
{v0.44}
{$IFNDEF CONSOLE}
procedure LanguageCheckForm(AForm: TForm);
  { Call usually from FormCreate event before assigning texts (so that
    the texts won't get replaced during FormActivate) }
{$ENDIF}
procedure StripLetters(var Buf; size: word);
  { Strips any hacky/carky from cz letters in char buffer buf (in Win1250 encoding) }
{/v0.44}
procedure LanguageDone;

function LanguageIsOn: boolean;


const
  FLanguage: TLanguage = nil;

implementation

function LanguageIsOn: boolean;
begin
  Result := FLanguage <> nil;
end;

{
constructor TStringObj.Create(const AText: string);
begin
  inherited Create;
  Text := AText;
end;}

{TLanguage}
{procedure TLanguage.SubClassWndProc(var Message: TMessage);
var
  i: integer;
  h: integer;
  f: TForm;
begin
  If Active then begin
    if (Message.Msg = WM_PARENTNOTIFY) then begin
      if LoWord(Message.WParam) = WM_CREATE then begin
        h := Message.LParam;
        for i := 0 to Screen.FormCount - 1 do begin
          f := Screen.Forms[i];
          if f.Handle = h then
            CheckForm(f);
        end;
      end;
    end;
  end;
  FOldWndMethod(Message);
end;

procedure TLanguage.UpdateOwner(AOwner: TComponent);
begin
  if AOwner <> nil then begin
    if Owner is TForm then begin
      if not Assigned(FOldWndMethod) then begin
        FOldWndMethod := TForm(Owner).WindowProc;
        TForm(Owner).WindowProc := SubClassWndProc;
      end;
    end;
  end else begin
    if (Owner <> nil) then begin
      if Assigned(FOldWndMethod) then begin
        if Owner is TForm then begin
          TForm(Owner).WindowProc := FOldWndMethod;
          FOldWndMethod := nil;
        end;
      end;
    end;
  end;
end;

procedure TLanguage.AfterConstruction;
begin
  inherited;
  UpdateOwner(Owner);
end;

procedure TLanguage.BeforeDestruction;
begin
  UpdateOwner(nil);
  inherited;
end;

const xx:integer = 0;}
function TLanguage.GetCheckedForms: TList;
begin
  if FCheckedForms = nil then
    FCheckedForms := TList.Create;
  Result := FCheckedForms;
end;

{v0.44}
function TLanguage.IsFormClassToIgnore(const AClassName: string): boolean;
const
  IgnoreList: array [0..7] of string = (
    'TULBrowseForm', 'TPickForm', 'TCheckListForm',
     {v0.46}'TMessageForm','TProgressForm'{/v0.46}{v0.47},'TULEditForm'{/v0.47}
     {v0.48},'TDBGridForm'{/v0.48} {v0.61},'TULBrowseModalForm'{/v0.61}
  );
var
  i: integer;
begin
  Result := false;
  for i := low(IgnoreList) to high(IgnoreList) do begin
    if AClassNAme = IgnoreList[i] then begin
      Result := true;
      exit;
    end;
  end;
end;
{/v0.44}

{$IFNDEF CONSOLE}
procedure TLanguage.ActiveFormChange(Sender: TObject);
begin
  CheckForms;
end;

procedure TLanguage.CheckForms;
var
  i: integer;
  l: TList;
  f: TForm;
begin
  {v0.44}
  if csDesigning in ComponentState then
    exit;
  if not Active then
    exit;
  {/v0.44}
  l := TList.Create;
  try
    { check for new forms }
    for i := 0 to Screen.FormCount - 1 do begin
      f := Screen.Forms[i];
      if {v0.47}(InspectAllForms or (f.Owner = Application)) and{/v0.47}
        (f.ClassName <> 'TForm') {v0.44} and (not IsFormClassToIgnore(f.ClassName)) {/v0.44} then
      begin
        l.Add(f);
        CheckForm(f);
      end;{ for TForm instances must be CheckForm called explicitely }
    end;

    {check for removed forms}
    i := 0;
    while i < CheckedForms.Count do begin
      f := TForm(CheckedForms.Items[i]);
      if l.IndexOf(f) < 0 then
        CheckedForms.Delete(i)
      else
        inc(i);
    end;
  finally
    l.Free;
  end;
end;
{$ENDIF}

procedure TLanguage.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then begin
    if AComponent = Self then begin
      {UpdateOwner(nil);}
    end;
  end else if (Operation = opInsert) then begin
    if AComponent = Self then begin
      {xx := 0;}
    end;
    {if (AComponent is TForm) then
      CheckForm(TForm(AComponent));}
  end;
end;

constructor TLanguage.Create(AOwner: TComponent);
begin
  if FLanguage <> nil then
    raise Exception.Create('Just one Language component can exist');
  inherited;
  FLanguage := Self;
  {FStrings := TStringList.Create;
  FStrings.Sorted := true;}
  FFileName := ChangeFileExt({v0.45}paramstr(0){/v0.45 Application.ExeName}, LngExt);
  Load;
end;

procedure TLanguage.SetFileName(const AFileName: string);
var
  n: string;
begin
  if csDesigning in ComponentState then begin
    FFileName := AFileName;
    exit;
  end;
  n := ExtractFilePath({v0.45}paramstr(0){/v0.45 Application.ExeName}) + ExtractFileName(AFileName);
  n := ChangeFileExt(n, LngExt);
  if not FileExists(n) then begin
    {$IFNDEF CONSOLE}
    ShowMessage(GetTxt({#}'Language file') + ' ' + n + ' ' + GetTxt({#}'not found, using current one.'));
    {$ENDIF}
    exit;
  end;
  if UpperCase(n) = UpperCase(FFileName) then
    exit;
  Save;
  FFileName := n;
  CheckedForms.Clear;{axisu}
  Load;
  {$IFNDEF CONSOLE}
  Check;{createform}
  {$ENDIF}
end;

function TLanguage.LoadIniFile: TMemIniFile;
begin
  if FIniFile <> nil then begin
    FIniFile.Free;
    FIniFile := nil;
  end;
  FIniFile := TMemIniFile.Create(FFileName);
  Result := FIniFile;
end;

procedure TLanguage.Loaded;
begin
  inherited;
  {$IFNDEF CONSOLE}
  Check;
  {$ENDIF}
end;

procedure TLanguage.Load;
{ var
  l: tstringlist;
  i: integer; }
begin
  LoadIniFile;
  {$IFDEF USESRC}
  if (not FileExists(FFileName)) or AlwaysScanSrcFiles then
    ScanSourceFiles;
  {$ENDIF}
  {
  if FileExists(FFileName) then begin
    l := TStringList.Create;
    try
      l.Sorted := true;
      l.LoadFromFile(CurLangFileName);
      for i := 0 to l.Count - 1 do begin
        FStrings.AddObject(l.Names[i], TStringObj.Create(l.Values[l.Names[i]]));
      end;
    finally
      l.Free;
    end;
  end;}
end;

procedure TLanguage.Save;
{var
  l: tstringlist;
  i: integer;}
begin
  {TMemIniFile}
  if FIniFile <> nil then begin
    FIniFile.UpdateFile;
    FIniFile.Free;
    FIniFile := nil;
  end;
  {
  if (FStrings = nil) or (FStrings.Count = 0) then
    exit;
  l := TStringList.Create;
  try
    for i := 0 to FStrings.Count - 1 do begin
      l.Add(FStrings[i] + '=' + TStringObj(FStrings.Objects[i]).Text);
    end;
    l.SaveToFile(CurLangFileName);
  finally
    l.Free;
  end;}
end;

destructor TLanguage.Destroy;
{var
  i: integer; tmethod timersu}
{$IFNDEF CONSOLE}
var
  t, t2: TNotifyEvent;
{$ENDIF}
begin
  if not (csDesigning in ComponentState) then begin
    {$IFNDEF CONSOLE}
    t := Screen.OnActiveFormChange;
    t2 := Self.ActiveFormChange;
    if CompareRec(t, t2, sizeof(TNotifyEvent)) = 0 then
      Screen.OnActiveFormChange := nil;
    {$ENDIF}
  end;
  FLanguage := nil;
  FCheckedForms.Free;
  FCheckedForms := nil;
  Save;
  {if FStrings <> nil then begin
    try
      Save;
      for i := 0 to FStrings.Count - 1 do
         FStrings.Objects[i].Free;
    except
    end;
    FStrings.Free;
  end;}
  inherited;
end;

procedure TLanguage.RegisterText(const ASection: string; const ATextName: string; const ATextValue: string);
      { Called for every text name/value found in components or passed to
        GetTxt function runtime (if in register mode). If not found in text
        storage, will add the item to the storage }
var
  sec, tn, tv: string;
  ov: string;
{  i: integer;}
const
  NonsenseString = '#@#x$%$$x.';
begin
  if Assigned(FOnRegisterText) then begin
    FOnRegisterText(ASection, tn, tv);
  end else begin
    sec := ASection;
    tn := ATextName;
    tv := ATextValue;
    if tn = '' then
      tn := tv;
    if sec = '' then
      sec := MiscSec;
    if (tn = '') then
      exit;
    ov := FIniFile.ReadString(ASection, tn, NonsenseString);
    if (NonsenseString = ov) { no value was in LNG yet }
       or
       ((ov = '') and (tv <> '')) { empty value was in LNG, but new value is <> '' }
    then
      FIniFile.WriteString(ASection, tn, tv + '@');
        { @ marks newly added item (too find it easily),
           during manual change this char should be removed }
    {
    if not FStrings.Find(tn, i) then begin
      FStrings.AddObject(tn, TStringObj.Create(tv));
    end;}
  end;
end;

function TLanguage.GetText(const ASection: string; const ATextName: string; const ADefValue: string): string;
      { If Active, then this method called from every GetTxt function,
        can replace ATextValue with the one found in the storage }
var
{  i: integer;}
  tn: string;
begin
  if ATextName = '' then
    tn := ADefValue
  else
    tn := ATextName;

  if FRegisterMode then begin
    RegisterText(ASection, tn, ADefValue);
  end;

  if Active then begin
    if Assigned(FOnGetText) then begin
      Result := FOnGetText(ASection, ATextName, ADefValue);
    end else begin
      Result := FIniFile.ReadString(ASection, tn, ADefValue);
      if (Result <> '') and (Result[Length(Result)] = '@') then
        SetLength(Result, Length(Result) - 1);
      {
      if FStrings.Find(tn, i) then begin
        Result := TStringObj(FStrings.Objects[i]).Text;
      end else begin
        Result := ADefValue;
      end;
      }

    end;
  end else begin
    Result := ADefValue
  end;
end;

{$IFNDEF CONSOLE}
procedure TLanguage.CheckForm(AForm: TForm);
begin
  if CheckedForms.IndexOf(AForm) < 0 then begin {beginupdate}
    CheckComponent(AForm.ClassName, '', AForm);
    CheckedForms.Add(AForm);
  end;
end;

procedure TLanguage.Check;
var
  i: integer;
  c: TComponent;
{  n: string;}
  orm: boolean;
begin
  if csDesigning in ComponentState then
    exit;
  orm := RegisterMode;{messages}
  RegisterMode := true;
  Active := true;
  try
    for i := 0 to Application.ComponentCount - 1 do begin
      c := Application.Components[i];
      if not (c is TForm) then
        continue;
      CheckForm(TForm(c));
    end;
  finally
    RegisterMode := orm;
  end;
  Screen.OnActiveFormChange := ActiveFormChange;
end;
{$ENDIF}

procedure TLanguage.CheckComponent(const ASection: string; const APathName: string; c: TComponent);
var
  ti: PTypeInfo;
  td: PTypeData;
  pl: TPropList;
  pi: PPropInfo;
  j: integer;
  n: string;
  v: string;
  nv: string;
  ch: TComponent;
  {v0.45}
  ignoreChilds:boolean;
  {v0.45}
begin
  {v0.45}
  ignoreChilds := false;
  {v0.45}
  if pos(ExcludePrefix, c.Name) = 1 then
    exit;

  {scan c properties}
  ti := PTypeInfo(c.ClassInfo);
  if ti = nil then
    exit;
  td := PTypeData(GetTypeData(ti));
  if td = nil then
    exit;
  GetPropInfos(ti, @pl);
  for j := 0 to td^.PropCount - 1 do begin
    pi := pl[j];
    if pi = nil then
      break;
    if pi^.Name = 'Name' then
      continue;
    if pos(ExcludePrefix, pi^.Name) = 1 then
      continue;

    case pi^.PropType^^.Kind of

      tkString, tkLString, tkWString: begin

         if not FAllTextProps then begin
           if (pos('Caption', pi^.Name) = 0) and (pos('Hint', pi^.Name) = 0)
             and (pos('Title', pi^.Name) = 0) and (pos('Label', pi^.Name) = 0)
           then
             continue;
         end;

         if APathName = '' then
           n := pi^.Name
         else
           n := APathName + '.' + pi^.Name;
         v := GetStrProp(c, pi);
         nv := GetText(ASection, n, v);
         if nv <> v then
           SetStrProp(c, pi, nv);
       end;
    end;

  end;
  {/scan c properties}

  {special cases}
  {$IFNDEF CONSOLE}
  if (c is TCustomMemo) then begin
    for j := 0 to TCustomMemo(c).Lines.Count-1 do
    begin
      n  := APathName + '.Lines[' + IntToStr(j) + ']';
      v := TCustomMemo(c).Lines[j];
      nv := GetText(ASection, n, v);
      if nv <> v then
        TCustomMemo(c).Lines[j] := nv;
    end;
  end;

  if (c is TRadioGroup) then begin
    for j := 0 to TRadioGroup(c).Items.Count - 1 do
    begin
      n := APathName + '.Items[' + IntToStr(j) + ']';
      v := TRadioGroup(c).Items[j];
      nv := GetText(ASection, n, v);
      if nv <> v then
        TRadioGroup(c).Items[j] := nv;
    end;
    {v0.45}
    ignoreChilds:= true;
    {v0.45}
  end;

  if (c is TComboBox) then begin
    for j := 0 to TComboBox(c).Items.Count - 1 do
    begin
      n := APathName + '.Items[' + IntToStr(j) + ']';
      v := TComboBox(c).Items[j];
      nv := GetText(ASection, n, v);
      if nv <> v then
        TComboBox(c).Items[j] := nv;
    end;
  end;
  {$ENDIF}
  {/special cases}

  {child components}
  {v0.45}
  if not ignoreChilds then
  {/v0.45}
  for j := 0 to c.ComponentCount - 1 do begin
    ch := c.Components[j];
    n := ch.Name;
    if n = '' then
      n := n + ch.ClassName;
    if APathName <> '' then
      n := APathName + '.' + n;
    CheckComponent(ASection, n, ch);
  end;
  {/child components}
end;


{$IFDEF USESRC}
procedure TLanguage.DoSrcLine(const ALine: string; var AResult: TDoLineResult);
var
  i: integer;
  line: string;
begin
  line := ALine;
  i := pos(SourceTextTag, line);
  if i <> 0 then begin
    line := copy(line, i + length(SourceTextTag), length(line));
    i := pos('''', line);
    if i > 0 then begin
      line := copy(line, i + 1, length(line));
      i := pos('''', line);
      if i > 0 then begin
        line := copy(line, 1, i - 1);
        RegisterText(MiscSec, '', line);
      end;
    end;
  end;
end;

procedure TLanguage.ScanSourceFiles;{moduutl}
var
  fs: TFileScanner;
begin
  fs := TFileScanner.Create(Self);
  try
    fs.Dir := ExtractFileDir({v0.45}paramstr(0){/v0.45 Application.ExeName});
    fs.Mask := '*.pas';
    fs.Recursive := true;
    fs.OnDoLine  := DoSrcLine;
    fs.Scan;
  finally
    fs.Free;
  end;
end;
{$ENDIF}

procedure TLanguage.SetLanguageName(const AName: string);
begin
  SetFileName(AName);
end;

function TLanguage.GetLanguageName: string;
begin
  Result := Uppercase(ChangeFileExt(ExtractFileName(FFileName), ''));
end;
{/TLanguage}

function GetTxt(const AText: string): string;
begin
  if FLanguage = nil then
    Result := AText
  else
    Result := FLanguage.GetText(MiscSec,'', AText);
end;

{v0.44}
{$IFNDEF CONSOLE}
procedure LanguageCheckForm(AForm: TForm);
  { Call usually from FormCreate event before assigning texts (so that
    the texts won't get replaced during FormActivate) }
begin
  if FLanguage <> nil then
    FLanguage.CheckForm(AForm);
end;
{$ENDIF}
{/v0.44}

{$IFNDEF CONSOLE}
procedure LanguageCheckForms;
begin
  if FLanguage <> nil then
    FLanguage.CheckForms;
end;
{$ENDIF}

procedure LanguageInit(const AFileName: string);
begin
  if FLanguage = nil then
    FLanguage := TLanguage.Create({v0.45}{$IFNDEF CONSOLE}Application{$ELSE}nil{$ENDIF});
  if AFileName <> '' then
    FLanguage.FileName := AFileName
  else begin
    {$IFNDEF CONSOLE}
    FLanguage.Check;
    {$ENDIF}
  end;
end;

procedure LanguageDone;
begin
  FLanguage.Free;
  FLanguage := nil;
end;

{v0.44}
procedure StripLetters(var Buf; size:word);
  { Strips any hacky/carky from cz letters in char buffer buf (in Win1250 encoding) }
  {country}
var
  p:array[0..pred(MaxWord)] of char absolute Buf;
  i:word;
  b:char;
begin
  if size = 0 then
    exit;
  for i := 0 to size - 1 do begin
    b := p[i];
    case p[i] of
      '' : b := 'a';
      '': b := 'A';
      '': b := 'c';
      '': b := 'C';
      '': b := 'd';
      '': b := 'D';
      '': b := 'e';
      '': b := 'E';
      '': b := 'e';
      '': b := 'E';
      '': b := 'i';
      '': b := 'I';
      '': b := 'n';
      '': b := 'N';
      '': b := 'o';
      '': b := 'O';
      '': b := 'r';
      '': b := 'R';
      '': b := 's';
      '': b := 'S';
      '': b := 't';
      '': b := 'T';
      '': b := 'u';
      '': b := 'U';
      '': b := 'u';
      '': b := 'U';
      '': b := 'y';
      '': b := 'Y';
      '': b := 'z';
      '': b := 'Z';
    end;
    p[i] := b;
  end;
end;
{/v0.44}

procedure Register;
begin
  RegisterComponents('NonVis', [TLanguage]);
end;

end.
