unit RF10AXLType;
interface
(*

  2004-06-05
  Information, types about device:

  Shimadzu Spectrofluorometric Detector RF-10AXL
  (for Shimadzu high performance liquid-chromatograph,
  ROM Ver.No. 3.20 or later)


1. Connecting external equipment

  Communicates using RS232. Uses 9 pin connector (only connected
  pins shown):

  RF (RF-10XLT side):
  pin	signal-name	function

  1	FG		Frame grounding
  2	TXD		Output data line
  3	RXD		Input data line
  7     SG		Signal grounding


  PC side:
  pin	signal-name	where to connect

  2	RXD 		connect to RF TXD
  3	TXD		connect to RF RXD
  4     DTR		connect to PC 6
  5     SG		connect to RF SG
  6	DSR		connect to PC 4
  7	RTS		connect to PC 8
  8	CTS		connect to PC 7



2. Setting parameters and activating remote control mode

    - press [func] (or [shift] + [func]) key repeatedly, till the display
      indicates the command "RS232C"
    - press [Enter] key to display  command for setting parameters for RS-232C
    - press [func] key repeatedly, the display indicates the commands in the
      following order:
        - activating remote control mode
            "CONNECT"
            - press [Enter] key, then RF is controlled from the external
              equipment via RS-232C interface. Then in this mode:
                - display then indicates excitation and emission wavelength
                  and emission intensity
                - RF outputs the analog signals of the emission intensity
                - any key input is rejected

        - setting baud rate
            "BAUD"
            Enter the desired value (i.e. 1..6) with the numeric key and press
            [Enter] key:

            RFBaudRates: array[1..6] of integer =
             (300, 600, 1200, 2400, 4800, 9600);

            or

            value = BaudRate div 300

        - setting data word size (value 7 or 8)
            "DATA BITS"

            RFDataBits = array[7..8] of integer = (7, 8);

        - setting parity (0..2)
            "PARITY"

             RFParities = array[0..2] of string = ('no parity', 'odd parity',
               'even parity');

        - setting stop bits
            "STOP BIT"

             RFStopBits = array[0..2] of string = ('no stop bit', '1 stop bit',
               '2 stop bits');

3. Transmission procedures and data format

    - all data used are in ASCII

    - for transmission the following 6 control codes are used:

      ENQ = #$05; // requests start of communication or requests response

      ACK = #$06; // positive acknowledge code. After corerct code or correct
                    text is received, this code is returned
      NAK = #$15; // negative acknowledge code. After incorrect code is received
                    or text is not eceived correctly, this code is returned
      STX = #$02; // indicates start of transmitted text

      ETX = #$03; // indicates end of transmitted text

      EOT = #$04; // indicates end of data transmission or communication


    - data format:

      [STX][Text][ETX][Sum]
      Text - the commands or measured data transmitted, 128 bytes max.
      Sum  - check sum for detection error in transmition, sums all bytes of
             Text plus EXT, then anded with $7F:
      see TRFData


    - file format:

      [File header][Spectrum data]
         (File transmitted from RF-10AXL to the external system by Command [FR].)

      [File header]:
         (Transmitted in one RFTxt)
         see TRFFileHead


    - transmission control procedures:

        - sender sends ENQ = requests receiver to listen for data
        - sender waits for ACK from receiver
        - if no ACK came, repeats sending ENQ
        - if ACK came, sender sends data
        - sender waits for ACK from receiver (confirming data received ok), or
          for NAK (reporting error)
        - if ACK came, sender sends EOT (notifying receiver no more data will
          be sent}
        - if NAK came, sender resends the data
        - if EOT came from receiver, then receiver can not fullfill command,
          sender will stop transmission.

4. Commands
    see cmXXXX section





*)

const
  ENQ = #$05; // requests start of communication or requests response

  ACK = #$06; // positive acknowledge code. After corerct code or correct
              //  text is received, this code is returned
  NAK = #$15; // negative acknowledge code. After incorrect code is received
              //  or text is not eceived correctly, this code is returned
  STX = #$02; // indicates start of transmitted text
                   
  ETX = #$03; // indicates end of transmitted text

  EOT = #$04; // indicates end of data transmission or communication

type

  TRFTxt = string[128];

  TRFData = record
    start: char; // always = STX, if data correct
    txt: TRFTxt;  // data to be tranmitted/received
    stop: char;  // always = ETX, if data correct
    sum: byte;   // checksum
  end;


const
  RFDeviceName = 'RF10AXL';
  {stXXXX Spectrum types}
  stExcitation = 1;
  stEmission = 2;
  {/stXXXX}

  {ssXXXX Scanning speeds}
  ssSuper = 1;
  ssFast = 2;
  ssMedium = 3;
  ssSlow = 4;
  {/ssXXXX}

  {seXXXX Sensitivities }
  seHigh = 0;
  seMed = 1;
  seLow = 2;
  {/seXXXX}

type
  TSpectrumType = stExcitation..stEmission;
  TScanningSpeed = ssSuper..ssSlow;
  TSensitivity = seHigh..seLow;

  TRFInt = array[0..2] of char; // some integer values transmitted in three chars (string)
  TWavelength = TRFInt;
  TRFLimit = array[0..4] of char;  // limits transmitted in 5 chars (string)

  TRFFileHead = packed record
    unused: char; // unused
    space1: char; // #$20
    SpectrumType: TSpectrumType; // 1 .. excitation, 2 .. emission
    space2: char; // #$20
    StargintWavelength: TWavelength; // for scanning
    space3: char;
    EndingWaveLength: TWavelength; // for scanning
    space4: char;
    ExcitationWavelength: TWavelength; // for emission spectrum excitation
         // wavelength is entered. For excitation spectrum 0 is entered.
    space5: char;
    EmissionWavelength: TWavelength; // For excitation spectrum, emission
         // wavelength is entered. For emission spectrum 0 is entered.
    space6: char;
    NumberOfData: TRFInt; // Number of data in file
    space7: char;
    UpperLimit: TRFLimit; // Upper limit of ordinate axis 10000. Upper
          // limit of measurement range multiplied by 10.
    space8: char;
    LowerLimit: TRFLimit; // Lower limit of ordinate axis -1000. Lower
          // limit of measurement range mutiplied by 10.
    space10: char;
    ScanningSpeed: TScanningSpeed;
    space11: char;
    Sensitivity: TSensitivity;
  end;

const
  {cmXXXX commands}
  cmWX = 'WX'{par}; // modify current excitation wavelength
                    // par=0,200-900; nm, in 1 nm increments; e.g.: "WX350"

  cmWM = 'WM'{par}; // modify current emission wavelength
                    // par=0,200-900; nm, in 1 nm increments; e.g.: "WM500"

  cmC = 'C'{par};   // set time constant
                    // par=1,2,3,4 (0.1 sec, 0.5 sec, 1.5 sec, 4 sec); e.g.: "C3"

  cmG = 'G'{par};   // set sensitivity (SENS)
                    // par=0,1,2 (HIGH, MED, LOW); e.g.: "G1"

  cmH = 'H'{par};   // select output mode (RATIO)
                    // par=0,1,2 (output excitation light energy, apply light source correction, output emission intensity not corrected);
                    // e.g.: "H1"

  cmI = 'I'{par};   // set amplification (GAIN) of signals
                    // par=1,2,3 (1x, 4x, 16x); e.g.: "I1"

  cmR = 'R'{par};   // set range of recorder output terminals
                    // par=1-10 (1, 1/2, 1/4, 1/8, 1/16, 1/32, 1/64, 1/128, 1/256, 0 mV);
                    // e.g. "R3"

  cmES = 'ES';      // short-circuits EVENT OUT terminals for 1 sec.

  cmEM = 'EM';      // add event mark to analog output

  cmZ = 'Z';        // executes auto zero function

  cmA = 'A'{par};   // selects operation mode when START terminals are short-circuited
                    // par=1,2 (auto zero, auto zero/mark);
                    // execute Auto zero adjustment, execute Auto zero adjustment and mark chromatogram and short the EVENT OUT terminal;
                    // e.g.: "A1"

  cmM = 'M'{par};   // selects type of spectrum (EX - excitation, or EM - emission) to be measured
                    // par=1,2 (EX, EM)

  cmKX = 'KX'{par}; // set scanning range of excitation monochromator
                    // par=200-900 200-900; in 1 nm units, starting wl should be shorter than ending
                    // e.g.: "KX300 500"

  cmKM = 'KM'{par}; // set scanning range of emission monochromator
                    // par=200-900 200-900;
                    // e.g.: "KM350 550"

  cmJ = 'J'{par};   // set scanning speed
                    // par=1,2,3,4 (SUPER, FAST, MEDIUM, SLOW);
                    // 3000 nm/min, 600 nm/min, 120 nm/min, 24 nm/min
                    // e.g.: "J2"

  cmB = 'B'{par};   // select ON/OFF of analog output during spectrum measurement
                    // par=1,2 (ON, OFF)
                    // e.g.: "B2"

  cmFI = 'FI'{par}; // stores data on measured spectrum in file
                    // par=1,2,5
                    // After scanning, the measured spectrum data is storted
                    // to file 0, this command copies the data to specified other
                    // file. File 5 is background data for differential spectrum processing.
                    // e.g.: "FI1" stores data on measured spectrum to file 1

  cmFR = 'FR'{par}; //* call spectrum file
                    // par=0,1,2,5
                    // transmits data stored in spectrum file to the external system
                    // using communication protocol described above

  cmFD = 'FD'{par}; // delete spectrum file
                    // par=0,1,2,5
                    // delete data in specified spectrum file (clear the file)
                    // e.g.: "FD1"

  cmL = 'L';        // execute spectrum measurement
                    // Starte measuring of the spectra. Command "L" or EOT can
                    // be sent to interrupt the measurement.

  cmS = 'S';        // calculate the difference between the measured spectrum and the
                    // stored background spectrum (i.e.: file 0 = file 0 - file 5)

  cmV = 'V';        //* requests current emission intensity
                    // after receiving this command RF sends its current emission
                    // intensity usin the protocol described above. The format of
                    // transmitted data is: "  250.00" (two space plus value), if
                    // the intensity is 250.00

  cmQ = 'Q'{par};   //* clear error or request code
                    // par=1,2 (clear error, request error code)
                    // e.g.: "Q2" request code of occurred error. After receiving
                    // this command RF sends error as three digit ASCII code,
                    // or of no error occurred sends "000". See RFErrors

  //* - data is sent back from RF-10AXL.
  {/cmXXXX}

type
  TRFErrorCode = byte;
  TRFErrorDesc = PChar;
  TRFErrorRec = record
    code: TRFErrorCode;
    desc: TRFErrorDesc;
  end;

const
  RFErrorCount = 12;
  RFErrors: array[0..RFErrorCount - 1] of TRFErrorRec = (
    (code:  1; desc: 'In scanning range setting, starting wavelength exceeds ending wavelength'),
    (code:  2; desc: 'Numeric value was tried to be set beyond allowable range'),
    (code:  4; desc: 'File is empty'),
    (code:  5; desc: 'Shift + CE key has interrupted execution'),
    (code:  9; desc: 'Auto zero function cannot be execute'),
    (code: 10; desc: 'Type of two spectrum (EX or EM) is not same during differential spectrum processing'),
    (code: 11; desc: 'Wavelength range of two spectrum is not same during differential spectrum processing'),
    (code:132; desc: 'Wavelength in excitation monochromator is not correct'),
                    // preform initial setting again. If the same error occurs again, contact Shimadzu
    (code:133; desc: 'Wavelength is emission monochromator is not correct'),
    (code:144; desc: 'Command is invalid'),
    (code:254; desc: 'Errors which can be reset with the CE key have occured'),
    (code:255; desc: 'Errors which can not be reset have occured')
  );

implementation
end.
